<?php
/*
Plugin Name: WooCommerce WhatsApp Notifications
Plugin URI: https://maxbot.live/
Description: Seamlessly integrate WhatsApp messaging with your WooCommerce store. Notify customers via WhatsApp after phone verification, and automatically send coupon reminders to users with incomplete purchases.
Version: 1.2.6
Author: Maxbot Technologies
Author URI: https://maxbot.live/
License: GPLv3
License URI: https://www.gnu.org/licenses/gpl-3.0.html
*/
/*
Copyright (C) 2025 Maxbot Technologies

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program. If not, see <https://www.gnu.org/licenses/>.
*/



include_once plugin_dir_path(__FILE__) . '/php/class-whatsapp-notifier.php';
include_once plugin_dir_path(__FILE__) . '/php/class_whatsapp_users.php';


if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}


function enqueue_custom_scripts()
{
    if (!is_admin()) {
        wp_enqueue_script('jquery');
        wp_enqueue_script('jquery-ui-dialog');

        // Enqueue the jQuery UI stylesheet from a CDN
        wp_enqueue_style('jquery-ui-css', 'https://code.jquery.com/ui/1.12.1/themes/base/jquery-ui.css');
    }
}
add_action('wp_enqueue_scripts', 'enqueue_custom_scripts');

add_action('admin_enqueue_scripts', 'whatsapp_qr_enqueue_scripts');
function whatsapp_qr_enqueue_scripts()
{
    $random_str = md5(uniqid(rand(), true));
    wp_enqueue_script('whatsapp-qr-script', plugin_dir_url(__FILE__) . 'js/whatsapp-qr.js?v=' . $random_str, ['jquery'], null, true);
    wp_localize_script('whatsapp-qr-script', 'whatsappQR', [
        'ajax_url' => admin_url('admin-ajax.php'),
        'x_key' => get_option('whatsapp_x_key', ''),
        'site_url' => $_SERVER['HTTP_HOST'],
        'nonce' => wp_create_nonce('generate_qr_nonce')
    ]);
}

function wc_custom_popup_scripts()
{
    // Load intl-tel-input CSS
    wp_enqueue_style(
        'intl-tel-input-css',
        'https://cdnjs.cloudflare.com/ajax/libs/intl-tel-input/25.3.1/build/css/intlTelInput.min.css',
        array(),
        '25.3.1'
    );

    // Load intl-tel-input JS
    wp_enqueue_script(
        'intl-tel-input-js',
        'https://cdnjs.cloudflare.com/ajax/libs/intl-tel-input/25.3.1/build/js/intlTelInput.min.js',
        array('jquery'),
        '25.3.1',
        true
    );

    // Load the utils script (required for formatting and validation)
    wp_enqueue_script(
        'intl-tel-utils-js',
        'https://cdn.jsdelivr.net/npm/intl-tel-input@25.3.1/build/js/utils.js',
        array(),
        '25.3.1',
        true
    );

    $random_str = md5(uniqid(rand(), true));
    wp_enqueue_script(
        'wc-custom-popup-script',
        plugins_url('/js/custom-popup.js?v=' . $random_str, __FILE__),
        array('jquery', 'intl-tel-input-js'),
        gmdate('Y-m-d H:i:s'),
        true
    );

    wp_localize_script('wc-custom-popup-script', 'wcCustomPopupData', array());
}
add_action('wp_enqueue_scripts', 'wc_custom_popup_scripts');

add_action('init', 'handle_whatsapp_parameters');

function handle_whatsapp_parameters()
{
    session_start();
    // Check if the 'action' parameter is set to 'fromWhatsapp'
    if (isset($_GET['action']) && $_GET['action'] === 'fromWhatsapp') {
        // Ensure the 'phone' parameter is also set
        if (isset($_GET['phone'])) {
            $phone = sanitize_text_field($_GET['phone']);
            $couponCode = sanitize_text_field($_GET['couponCode']);
            $user = WhatsAppUsers::get_by_phone($phone);
            $user_verify_phone_time = gmdate('Y-m-d H:i:s');
            $_SESSION["whatsapp_phone"] = $phone;
            $_SESSION["created_user_id"] = $user->id;
            $_SESSION["user_verify_phone_time"] = $user_verify_phone_time;
            $_SESSION['dontShowPopup'] = true;

            if (WC()->cart->is_empty()) {
                WhatsAppUsers::create_or_update($phone, [
                    'phone_verified'    => 1,
                    'user_verify_phone_time' => $user_verify_phone_time,
                    'coupon_code' => $couponCode,
                ]);
                $_SESSION['couponCode'] = $couponCode;
            } else {
                WC()->cart->add_discount($couponCode);
                WhatsAppUsers::create_or_update($phone, [
                    'phone_verified'    => 1,
                    'user_verify_phone_time' => $user_verify_phone_time,
                    'coupon_added_to_cart' => 1,
                    'coupon_code' => $couponCode,
                ]);
            }
            add_action('wp_footer', 'my_custom_alert_script');
        }
    }
}

add_action('woocommerce_add_to_cart', 'after_adding_to_cart', 10, 6);
function after_adding_to_cart($cart_item_key, $product_id, $quantity, $variation_id, $variation, $cart_item_data)
{
    $fullPhoneNumber = $_SESSION["whatsapp_phone"] ?? null;
    if ($fullPhoneNumber != null) {
        $existUser = WhatsAppUsers::exists($fullPhoneNumber);
        if ($existUser) {
            $user = WhatsAppUsers::get_by_phone($fullPhoneNumber);
            $coupon_added_to_cart = $user->coupon_added_to_cart ?? 0;
            if (isset($_SESSION['couponCode']) && $_SESSION['couponCode'] != "" && $coupon_added_to_cart == 0) {
                WC()->cart->add_discount($_SESSION['couponCode']);
                unset($_SESSION['couponCode']);
                WhatsAppUsers::create_or_update($fullPhoneNumber, [
                    'coupon_added_to_cart' => 1,
                ]);
            }
        }
    }
}

function my_custom_alert_script()
{
    echo "<script type='text/javascript'>alert('The coupon is waiting for you inside the checkout');</script>";
}

function whatsapp_notification_add_cart_body_class($classes)
{
    if (function_exists('is_cart') && is_cart()) {
        $classes[] = 'whatsapp_notification-cart-page'; // Add this class if on the cart page
    }
    return $classes;
}

add_filter('body_class', 'whatsapp_notification_add_cart_body_class');

function whatsapp_notification_custom_override_checkout_fields($fields)
{
    session_start();
    $hide_phone_field = false;
    if (isset($_SESSION["whatsapp_phone"])) {
        $hide_phone_field = true;
    }

    if ($hide_phone_field) {
        $fields['billing']['billing_phone']['custom_attributes'] = array('readonly' => 'readonly');
        $fields['billing']['billing_phone']['default'] = $_SESSION["whatsapp_phone"]; // Replace with the desired default value
    }

    return $fields;
}
add_filter('woocommerce_checkout_fields', 'whatsapp_notification_custom_override_checkout_fields');

// Hook to add admin menu
add_action('admin_menu', 'whatsapp_notification_add_admin_menu');

function whatsapp_notification_add_admin_menu()
{
    add_menu_page('Whatsapp notification', 'Whatsapp notification', 'manage_options', 'whatsapp_notification_custom_plugin', 'whatsapp_notification_options_page');
    // Additional page to display the custom table
    add_submenu_page(
        'whatsapp_notification_custom_plugin', // Parent slug
        'Users verified phone', // Page title
        'Users verified phone', // Menu title
        'manage_options', // Capability
        'whatsapp_notification_custom_table', // Menu slug
        'whatsapp_display_custom_table' // Function to display the page content
    );

    add_submenu_page(
        'whatsapp_notification_custom_plugin', // Parent menu slug
        'Send Bulk WhatsApp Message', // Page title
        'Send Bulk WhatsApp Message', // Menu title
        'manage_options', // Capability
        'whatsapp_bulk_broadcast', // Menu slug
        'whatsapp_bulk_broadcast_page' // Function that renders the page
    );
}

function whatsapp_bulk_broadcast_page()
{
    global $wpdb;
    $table_name = $wpdb->prefix . 'whatsapp_newsletters_sent';

    // Handle message sending
    if (isset($_POST['send_broadcast']) && check_admin_referer('send_whatsapp_broadcast', 'whatsapp_broadcast_nonce')) {
        $message = sanitize_textarea_field($_POST['broadcast_message']);
        $campaign_name = 'Campaign_' . date('Ymd_His');
        $subscribers = WhatsAppUsers::get_all_newsletter_agreed();
        $options = get_option('whatsapp_notification_settings');
        $sender_phone = $options['whatsapp_phone_scan_qr_code'];

        echo "<div class='updated notice'><p>Broadcast initiated to " . count($subscribers) . " subscribers.</p></div>";

        foreach ($subscribers as $user) {
            WhatsAppNotifier::sendRequestCustomWhatsappApi($user->phone, $message, $sender_phone);
            log_whatsapp_newsletter($user->phone, $message, $campaign_name, 1, null);
        }

        echo "<div class='notice notice-success'><p>Campaign \"$campaign_name\" sent to " . count($subscribers) . " users.</p></div>";
    }

    // Form
?>
    <div class="wrap">
        <h1>Send Bulk WhatsApp Message</h1>
        <p>Send a WhatsApp message to all users who subscribed to your newsletter.</p>

        <form method="post" action="">
            <?php wp_nonce_field('send_whatsapp_broadcast', 'whatsapp_broadcast_nonce'); ?>
            <textarea name="broadcast_message" rows="5" cols="60" placeholder="Type your message here..." required></textarea><br><br>
            <input type="submit" name="send_broadcast" class="button button-primary" value="Send Message">
        </form>
    </div>
    <hr>

    <div class="wrap">
        <h2>Sent Campaigns</h2>
        <table class="widefat fixed striped">
            <thead>
                <tr>
                    <th>Campaign</th>
                    <th>Recipients</th>
                    <th>Last Sent</th>
                </tr>
            </thead>
            <tbody>
                <?php
                $campaigns = $wpdb->get_results("
                    SELECT campaign_name, COUNT(*) as total, MAX(sent_at) as last_sent
                    FROM $table_name
                    GROUP BY campaign_name
                    ORDER BY last_sent DESC
                ");

                foreach ($campaigns as $campaign) {
                    $view_url = admin_url("admin.php?page=whatsapp_bulk_broadcast&campaign=" . urlencode($campaign->campaign_name));
                    echo "<tr>
                        <td><a href='$view_url'>" . esc_html($campaign->campaign_name) . "</a></td>
                        <td>" . esc_html($campaign->total) . "</td>
                        <td>" . esc_html($campaign->last_sent) . "</td>
                    </tr>";
                }

                if (empty($campaigns)) {
                    echo "<tr><td colspan='3'>No campaigns sent yet.</td></tr>";
                }
                ?>
            </tbody>
        </table>
    </div>
    <?php

    // Show recipients if a campaign is selected
    if (!empty($_GET['campaign'])) {
        $selected_campaign = sanitize_text_field($_GET['campaign']);
        $recipients = $wpdb->get_results($wpdb->prepare("
            SELECT phone, message, sent_at FROM $table_name
            WHERE campaign_name = %s
            ORDER BY sent_at ASC
        ", $selected_campaign));

        echo "<div class='wrap'><h2>Recipients for Campaign: " . esc_html($selected_campaign) . "</h2>";
        echo "<table class='widefat striped'>
            <thead><tr><th>Phone</th><th>Message</th><th>Sent At</th></tr></thead><tbody>";

        foreach ($recipients as $row) {
            echo "<tr>
                <td>" . esc_html($row->phone) . "</td>
                <td>" . esc_html($row->message) . "</td>
                <td>" . esc_html($row->sent_at) . "</td>
            </tr>";
        }

        echo "</tbody></table></div>";
    }
}



function whatsapp_display_custom_table()
{
    $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
    $users_per_page = 20;

    // Filters
    $verified = isset($_GET['verified']) ? sanitize_text_field($_GET['verified']) : '';
    $coupon_type = isset($_GET['coupon_type']) ? sanitize_text_field($_GET['coupon_type']) : '';

    // Display header and filter form
    echo '<div class="wrap"><h1>Users Verified Phone</h1>';

    echo '<form method="get" style="margin-bottom:20px;">';
    echo '<input type="hidden" name="page" value="' . esc_attr($_GET['page']) . '" />';

    echo '<label for="verified">Phone Verified:</label> ';
    echo '<select name="verified" id="verified">';
    echo '<option value="">All</option>';
    echo '<option value="1"' . selected($verified, '1', false) . '>Yes</option>';
    echo '<option value="0"' . selected($verified, '0', false) . '>No</option>';
    echo '</select> ';

    echo '<label for="coupon_type">Coupon Type:</label> ';
    echo '<select name="coupon_type" id="coupon_type">';
    echo '<option value="">All</option>';
    echo '<option value="verification"' . selected($coupon_type, 'verification', false) . '>Verification Coupon</option>';
    echo '<option value="cart"' . selected($coupon_type, 'cart', false) . '>Cart Coupon</option>';
    echo '<option value="added"' . selected($coupon_type, 'added', false) . '>Coupon Added to Cart</option>';
    echo '</select> ';

    echo '<input type="submit" class="button" value="Filter" />';
    echo '</form>';

    // Fetch data
    $filters = [
        'verified' => $verified,
        'coupon_type' => $coupon_type,
    ];

    $total_users = WhatsAppUsers::get_total($filters);
    $total_pages = ceil($total_users / $users_per_page);

    $users = WhatsAppUsers::get_all($current_page, $users_per_page, $filters);

    // Table header
    echo '<table class="widefat fixed" cellspacing="0">';
    echo '<thead><tr>
        <th>#</th>
        <th>Phone</th>
        <th>Created At</th>
        <th>Verified At</th>
        <th>Phone Verified</th>
        <th>Verification Coupon</th>
        <th>Cart Coupon</th>
        <th>Coupon Code</th>
        <th>Coupon Added</th>
    </tr></thead>';
    echo '<tbody>';

    $row_number = ($current_page - 1) * $users_per_page;

    foreach ($users as $user) {
        $row_number++;
        echo '<tr>';
        echo "<td>{$row_number}</td>";
        echo "<td>" . esc_html($user->phone) . "</td>";

        // Format created_at
        $created_at_display = '—';
        if ($user->created_at && $user->created_at !== '0000-00-00 00:00:00') {
            $dt = new DateTime($user->created_at, new DateTimeZone('UTC'));
            $dt->setTimezone(wp_timezone());
            $created_at_display = esc_html($dt->format('Y-m-d H:i:s'));
        }
        echo "<td>{$created_at_display}</td>";

        // Format verify_time
        $verify_time_display = '—';
        if ($user->user_verify_phone_time && $user->user_verify_phone_time !== '0000-00-00 00:00:00') {
            $dt = new DateTime($user->user_verify_phone_time, new DateTimeZone('UTC'));
            $dt->setTimezone(wp_timezone());
            $verify_time_display = esc_html($dt->format('Y-m-d H:i:s'));
        }
        echo "<td>{$verify_time_display}</td>";


        echo "<td>" . ($user->phone_verified ? 'Yes' : 'No') . "</td>";
        echo "<td>" . ($user->got_verification_coupon ? 'Yes' : 'No') . "</td>";
        echo "<td>" . ($user->got_cart_coupon ? 'Yes' : 'No') . "</td>";
        echo "<td>" . esc_html($user->coupon_code ?: '—') . "</td>";
        echo "<td>" . ($user->coupon_added_to_cart ? 'Yes' : 'No') . "</td>";
        echo '</tr>';
    }


    echo '</tbody></table>';

    // Pagination
    $page_links = paginate_links([
        'base' => add_query_arg('paged', '%#%'),
        'format' => '',
        'prev_text' => __('&laquo;', 'text-domain'),
        'next_text' => __('&raquo;', 'text-domain'),
        'total' => $total_pages,
        'current' => $current_page
    ]);

    echo '<div id="pagination" class="tablenav"><div class="tablenav-pages">' . $page_links . '</div></div>';
    echo '</div>';
}



function whatsapp_notification_options_page()
{ ?>
    <div class="wrap">
        <h2>Whatsapp notification</h2>
        <form action="options.php" method="post">
            <?php
            settings_fields('whatsapp_notification_page');
            do_settings_sections('whatsapp_notification_page');
            submit_button();
            ?>
        </form>
    </div>
<?php }

add_action('admin_init', 'whatsapp_notification_settings_init');

function whatsapp_notification_settings_init()
{
    // Register a new setting for the settings page.
    register_setting('whatsapp_notification_page', 'whatsapp_notification_settings');

    // Add section to the settings page.
    add_settings_section(
        'whatsapp_notification_section',
        __('WhatsApp Notification Settings', 'whatsapp_notification'),
        'whatsapp_notification_section_callback',
        'whatsapp_notification_page'
    );

    add_settings_field(
        'whatsapp_notification_initial_coupon_visible',
        __('Enable Popup for WhatsApp Coupon', 'whatsapp_notification'),
        'whatsapp_notification_initial_coupon_visible_render',
        'whatsapp_notification_page',
        'whatsapp_notification_section'
    );

    add_settings_field(
        'whatsapp_notification_initial_coupon_discount',
        __('Discount Percentage After Phone Verification', 'whatsapp_notification'),
        'whatsapp_notification_initial_coupon_discount_render',
        'whatsapp_notification_page',
        'whatsapp_notification_section'
    );

    add_settings_field(
        'whatsapp_cart_abandon_visible',
        __('Enable WhatsApp Reminder for Abandoned Carts', 'whatsapp_notification'),
        'whatsapp_cart_abandon_visible_render',
        'whatsapp_notification_page',
        'whatsapp_notification_section'
    );

    add_settings_field(
        'whatsapp_notification_time_after_cart_abandon',
        __('Delay Before Sending Abandoned Cart Message (minutes)', 'whatsapp_notification'),
        'whatsapp_notification_time_after_cart_abandon_render',
        'whatsapp_notification_page',
        'whatsapp_notification_section'
    );

    add_settings_field(
        'whatsapp_notification_cart_abandon_coupon_discount',
        __('Discount Percentage for Abandoned Cart Recovery', 'whatsapp_notification'),
        'whatsapp_notification_cart_abandon_coupon_discount_render',
        'whatsapp_notification_page',
        'whatsapp_notification_section'
    );

    add_settings_field(
        'whatsapp_order_status_to_ignore',
        __('Order Statuses That Prevent Abandoned Cart Reminder', 'whatsapp_notification'),
        'whatsapp_order_status_to_ignore_render',
        'whatsapp_notification_page',
        'whatsapp_notification_section'
    );

    add_settings_field(
        'whatsapp_phone_scan_qr_code',
        __('Connect WhatsApp Phone via QR Code', 'whatsapp_notification'),
        'whatsapp_phone_scan_qr_code_render',
        'whatsapp_notification_page',
        'whatsapp_notification_section'
    );
}

function whatsapp_notification_section_callback()
{
    echo __('', 'whatsapp_notification');
}

function whatsapp_order_status_to_ignore_render()
{
    $options = get_option('whatsapp_notification_settings');
    $selected = $options['whatsapp_order_status_to_ignore'] ?? [];

    $statuses = wc_get_order_statuses(); // returns array like ['wc-completed' => 'Completed', ...]
    foreach ($statuses as $status_key => $status_label) {
        $checked = in_array($status_key, $selected) ? 'checked' : '';
        echo "<label><input type='checkbox' name='whatsapp_notification_settings[whatsapp_order_status_to_ignore][]' value='$status_key' $checked> $status_label</label><br>";
    }
}


function whatsapp_notification_initial_coupon_visible_render()
{
    $options = get_option('whatsapp_notification_settings');
    $whatsapp_notification_initial_coupon_visible = $options['whatsapp_notification_initial_coupon_visible'] ?? 'false';
?>
    <input type='checkbox' <?php echo $whatsapp_notification_initial_coupon_visible == 'true' ? 'checked' : ''; ?> name='whatsapp_notification_settings[whatsapp_notification_initial_coupon_visible]'>
    <p class="description"><?php _e('Show a popup to collect user phone numbers in exchange for a WhatsApp discount coupon.', 'whatsapp_notification'); ?></p>
<?php
}

function whatsapp_notification_initial_coupon_discount_render()
{
    $options = get_option('whatsapp_notification_settings');
?>
    <input type='text' name='whatsapp_notification_settings[whatsapp_notification_initial_coupon_discount]' value='<?php echo esc_attr($options['whatsapp_notification_initial_coupon_discount']); ?>'>
    <p class="description"><?php _e('Set the discount percentage sent to the user after they verify their phone number via WhatsApp.', 'whatsapp_notification'); ?></p>
<?php
}

function whatsapp_cart_abandon_visible_render()
{
    $options = get_option('whatsapp_notification_settings');
    $whatsapp_cart_abandon_visible = $options['whatsapp_cart_abandon_visible'] ?? 'false';
?>
    <input type='checkbox' <?php echo $whatsapp_cart_abandon_visible == 'true' ? 'checked' : ''; ?> name='whatsapp_notification_settings[whatsapp_cart_abandon_visible]'>
    <p class="description"><?php _e('Send WhatsApp notifications to users who added items to their cart but didn’t complete the purchase.', 'whatsapp_notification'); ?></p>
<?php
}

function whatsapp_notification_time_after_cart_abandon_render()
{
    $options = get_option('whatsapp_notification_settings');
?>
    <input type='text' name='whatsapp_notification_settings[whatsapp_notification_time_after_cart_abandon]' value='<?php echo esc_attr($options['whatsapp_notification_time_after_cart_abandon']); ?>'>
    <p class="description"><?php _e('Number of minutes after cart abandonment before sending the discount coupon via WhatsApp.', 'whatsapp_notification'); ?></p>
<?php
}

function whatsapp_notification_cart_abandon_coupon_discount_render()
{
    $options = get_option('whatsapp_notification_settings');
?>
    <input type='text' name='whatsapp_notification_settings[whatsapp_notification_cart_abandon_coupon_discount]' value='<?php echo esc_attr($options['whatsapp_notification_cart_abandon_coupon_discount']); ?>'>
    <p class="description"><?php _e('Set the discount percentage offered to recover an abandoned cart.', 'whatsapp_notification'); ?></p>
    <?php
}


function whatsapp_phone_scan_qr_code_render()
{
    $options = get_option('whatsapp_notification_settings');



    $phone = $options['whatsapp_phone_scan_qr_code'];

    $session_status = call_whatsapp_api('https://gateway.ask-us.link/check_whatsapp_session', $phone);
    $connected = $session_status['error'] ? false : $session_status['data']['connected'];
    if ($connected) {
    ?>
        <div>
            <label><?php echo esc_attr($phone); ?> - ✅ Connected</label><br>
            <button type="button" id="disconnect_phone_button">Disconnect Phone</button>
            <div id="disconnect_status" style="margin-top: 10px;"></div>
        </div>


    <?php
    } else {
    ?>
        <div>
            <label for="whatsapp_phone">Enter WhatsApp Phone Number:</label><br>
            <input type="text" id="whatsapp_phone" value="<?php echo esc_attr($phone); ?>" />
            <button type="button" id="generate_qr_button">Generate QR Code</button>
        </div>
        <div id="qr_result" style="margin-top:10px;">
            <img id="qr_code" src="" alt="QR Code" style="display: none;" />
        </div>
    <?php
    }

    ?>
    <label style="word-break: break-all;"><?php echo json_encode($options) ?></label>
<?php
}



register_setting('whatsapp_notification_page', 'whatsapp_notification_settings', 'whatsapp_notification_settings_validate');

function whatsapp_notification_settings_validate($input)
{
    $new_input = array();

    if (isset($input['whatsapp_notification_time_after_cart_abandon'])) {
        $new_input['whatsapp_notification_time_after_cart_abandon'] = absint($input['whatsapp_notification_time_after_cart_abandon']);
    }

    if (isset($input['whatsapp_notification_cart_abandon_coupon_discount'])) {
        $new_input['whatsapp_notification_cart_abandon_coupon_discount'] = sanitize_text_field($input['whatsapp_notification_cart_abandon_coupon_discount']);
    }

    if (isset($input['whatsapp_notification_initial_coupon_discount'])) {
        $new_input['whatsapp_notification_initial_coupon_discount'] = sanitize_text_field($input['whatsapp_notification_initial_coupon_discount']);
    }

    // ✅ Always set to "true" or "false"
    $new_input['whatsapp_notification_initial_coupon_visible'] = isset($input['whatsapp_notification_initial_coupon_visible']) ? 'true' : 'false';
    $new_input['whatsapp_cart_abandon_visible'] = isset($input['whatsapp_cart_abandon_visible']) ? 'true' : 'false';

    if (isset($input['whatsapp_phone_scan_qr_code'])) {
        $new_input['whatsapp_phone_scan_qr_code'] = sanitize_text_field($input['whatsapp_phone_scan_qr_code']);
    }

    if (isset($input['whatsapp_order_status_to_ignore']) && is_array($input['whatsapp_order_status_to_ignore'])) {
        $new_input['whatsapp_order_status_to_ignore'] = array_map('sanitize_text_field', $input['whatsapp_order_status_to_ignore']);
    }


    return array_merge(get_option('whatsapp_notification_settings', []), $new_input);
}



function call_whatsapp_api($endpoint, $phone)
{
    $x_key = get_option('whatsapp_x_key', '');
    $json_payload = json_encode([
        'phone' => $phone,
        'ar_secret' => 'EoOCJJwQKBmdPdmFy4uLMhz1OCOPF64FeMxL1AXQO84THs9jeCJHjOJZtBjbah6L',
        'website_name' => $_SERVER['HTTP_HOST']
    ]);

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $endpoint);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $json_payload);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'X-Key: ' . $x_key,
    ]);

    $response = curl_exec($ch);
    $curl_error = curl_error($ch);
    curl_close($ch);

    if ($curl_error) {
        return ['error' => true, 'message' => $curl_error];
    } else {
        return ['error' => false, 'data' => json_decode($response, true)];
    }
}

add_action('wp_ajax_generate_whatsapp_qr', function () {
    check_ajax_referer('generate_qr_nonce', 'nonce');
    $phone = sanitize_text_field($_POST['phone']);
    $options = get_option('whatsapp_notification_settings');
    $options['whatsapp_phone_scan_qr_code'] = $phone;
    update_option('whatsapp_notification_settings', $options);
    $result = call_whatsapp_api('https://gateway.ask-us.link/qr-code-generator', $phone);
    $result['error'] ? wp_send_json_error($result['message']) : wp_send_json_success($result['data']);
});

add_action('wp_ajax_get_qr_from_server', function () {
    check_ajax_referer('generate_qr_nonce', 'nonce');
    $phone = sanitize_text_field($_POST['phone']);
    $result = call_whatsapp_api('https://gateway.ask-us.link/get_qr', $phone);
    $result['error'] ? wp_send_json_error($result['message']) : wp_send_json_success($result['data']);
});

add_action('wp_ajax_check_whatsapp_session_status', function () {
    check_ajax_referer('generate_qr_nonce', 'nonce');
    $phone = sanitize_text_field($_POST['phone']);
    $result = call_whatsapp_api('https://gateway.ask-us.link/check_whatsapp_session', $phone);
    $result['error'] ? wp_send_json_error($result['message']) : wp_send_json_success($result['data']);
});

add_action('wp_ajax_disconnect_whatsapp_phone', 'ajax_disconnect_whatsapp_phone');

function ajax_disconnect_whatsapp_phone()
{
    check_ajax_referer('generate_qr_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }

    $options = get_option('whatsapp_notification_settings', []);
    $phone = $options['whatsapp_phone_scan_qr_code'] ?? '';

    if ($phone) {
        call_whatsapp_api('https://gateway.ask-us.link/disconnect_whatsapp_session', $phone);
    }

    unset($options['whatsapp_phone_scan_qr_code']);
    update_option('whatsapp_notification_settings', $options);

    wp_send_json_success('Phone disconnected');
}



// CRON JOB
add_filter('cron_schedules', 'whatsapp_notification_cron_schedule');

function whatsapp_notification_cron_schedule($schedules)
{
    if (!isset($schedules['every_five_minutes'])) {
        $schedules['every_five_minutes'] = array(
            'interval' => 300,  // Correcting the interval to 300 seconds
            'display'  => __('Every Five Minutes', 'whatsapp_notification')
        );
    }
    return $schedules;
}


// Correcting the activation and deactivation hooks
register_activation_hook(__FILE__, 'whatsapp_notification_schedule_cron');
register_deactivation_hook(__FILE__, 'whatsapp_notification_deactivate_cron');

function whatsapp_notification_schedule_cron()
{
    if (!wp_next_scheduled('whatsapp_notification_cron_hook')) {
        wp_schedule_event(gmdate('Y-m-d H:i:s'), 'every_five_minutes', 'whatsapp_notification_cron_hook');
    }

    $options = get_option('whatsapp_notification_settings');
    if ($options['whatsapp_x_key'] == "") {
        $xkey = bin2hex(random_bytes(20));
        $options['whatsapp_x_key'] = $xkey;
        update_option('whatsapp_notification_settings', $options);
    }
}

function whatsapp_notification_deactivate_cron()
{
    $timestamp = wp_next_scheduled('whatsapp_notification_cron_hook');
    if ($timestamp) {
        wp_unschedule_event($timestamp, 'whatsapp_notification_cron_hook');
    }
}

add_action('whatsapp_notification_cron_hook', 'whatsapp_notification_cron_task_function');

function whatsapp_notification_cron_task_function()
{
    $users = WhatsAppUsers::get_all_not_got_cart_coupon(); // Get all users with verified phone numbers
    $options = get_option('whatsapp_notification_settings');
    $whatsapp_notification_time_after_cart_abandon = $options['whatsapp_notification_time_after_cart_abandon'] ?? '30';  // Using the null coalescing operator to avoid undefined index notice.
    $whatsapp_notification_cart_abandon_coupon_discount = $options['whatsapp_notification_cart_abandon_coupon_discount'] ?? '10';  // Using the null coalescing operator to avoid undefined index notice.
    $phone = $options['whatsapp_phone_scan_qr_code'];
    foreach ($users as $user) {
        $fullPhoneNumber = $user->phone;
        $country_code = $user->country_code;
        $user_verify_phone_time = $user->user_verify_phone_time ? strtotime($user->user_verify_phone_time) : gmdate('Y-m-d H:i:s'); // Get the verification time or use the current time if not set
        $verify_datetime = new DateTime('@' . $user_verify_phone_time);

        if (has_time_passed($verify_datetime, $whatsapp_notification_time_after_cart_abandon) && !has_user_made_order_since($fullPhoneNumber, $country_code, $verify_datetime)) {
            // Send the WhatsApp notification
            // Update the user meta to indicate that the notification has been sent
            $new_coupon_code = WhatsAppNotifier::generateCoupon($whatsapp_notification_cart_abandon_coupon_discount);

            $message = "We saw that you did not complete the purchase and wanted to offer you a discount coupon of " . $whatsapp_notification_cart_abandon_coupon_discount . "%. The coupon number is: " . $new_coupon_code . " The coupon is valid for 10 hours only. You can enter the site by clicking on the link " . $_SERVER['HTTP_HOST'] . "/?action=fromWhatsapp&sub=cart_abandon&country_code=" . $country_code . "&phone=" . $fullPhoneNumber . "&couponCode=" . $new_coupon_code . "";
            WhatsAppNotifier::sendRequestCustomWhatsappApi($fullPhoneNumber, $message, $phone);
            WhatsAppUsers::create_or_update($user->phone, [
                'got_cart_coupon' => 1,
                'coupon_code' => $new_coupon_code,
            ]);
        }
    }
}

function has_time_passed($verify_datetime, $minutes)
{
    $current_datetime = new DateTime('now', new DateTimeZone('UTC'));  // Ensure timezone consistency
    $interval = $current_datetime->diff($verify_datetime);
    $minutes_passed = $interval->i + ($interval->h * 60) + ($interval->days * 24 * 60);  // Convert days and hours to minutes and add
    return $minutes_passed >= $minutes; // Return true if the specified minutes have passed
}

function has_user_made_order_since($phone, $country_code, $since_date)
{
    $phone_no_CC = "0" .  str_replace($country_code, "", $phone);
    $formatted_date = $since_date->format('Y-m-d H:i:s');

    $options = get_option('whatsapp_notification_settings');
    $order_statuses = $options['whatsapp_order_status_to_ignore'] ?? ['wc-completed']; // default fallback

    $orders = wc_get_orders(array(
        'limit' => -1,
        'status' => $order_statuses,
        'date_query' => array(
            array(
                'after'     => $formatted_date,
                'inclusive' => true,
            ),
        ),
    ));

    $matched_orders = [];

    foreach ($orders as $order) {
        if ($order->get_billing_phone() === $phone_no_CC) {
            $matched_orders[] = $order;
        }
    }

    // Now $matched_orders contains all orders by phone number


    return count($matched_orders) > 0;
}


add_action('woocommerce_after_checkout_form', 'init_iti_on_billing_phone_field');
function init_iti_on_billing_phone_field()
{
?>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const input = document.querySelector('#billing_phone');
            const form = document.querySelector('form.checkout'); // always target the real WC form

            if (!input || !form || typeof window.intlTelInput !== 'function') {
                console.warn("intlTelInput is not ready yet.");
                return;
            }

            const iti = window.intlTelInput(input, {
                initialCountry: "auto",
                geoIpLookup: function(callback) {
                    fetch("https://ipapi.co/json")
                        .then((res) => res.json())
                        .then((data) => callback(data.country_code))
                        .catch(() => callback("us"));
                },
                loadUtils: () =>
                    import(
                        "https://cdn.jsdelivr.net/npm/intl-tel-input@25.3.1/build/js/utils.js"
                    ),
            });

            console.log('intlTelInput initialized');

            // Create and append hidden inputs
            const hiddenInput = document.createElement('input');
            hiddenInput.type = 'hidden';
            hiddenInput.name = 'whatsapp_full_number';
            form.appendChild(hiddenInput);

            const hiddenInputCountryCode = document.createElement('input');
            hiddenInputCountryCode.type = 'hidden';
            hiddenInputCountryCode.name = 'whatsapp_country_code';
            form.appendChild(hiddenInputCountryCode);

            // Update functions
            const updateHidden = () => {
                if (typeof iti.getNumber === 'function') {
                    hiddenInput.value = iti.getNumber();
                    console.log('Full number updated:', hiddenInput.value);
                }
            };

            const updateHiddenCountryCode = () => {
                if (typeof iti.getSelectedCountryData === 'function') {
                    hiddenInputCountryCode.value = iti.getSelectedCountryData().dialCode;
                    console.log('Country code updated:', hiddenInputCountryCode.value);
                }
            };

            // Ensure values update during input
            ['blur', 'change', 'keyup'].forEach(event => {
                input.addEventListener(event, () => {
                    updateHidden();
                    updateHiddenCountryCode();
                });
            });

            // Force update before submitting form (critical fix!)
            form.addEventListener('submit', () => {
                updateHidden();
                updateHiddenCountryCode();
            });
        });
    </script>

    <?php
}

add_action('wp_head', function () {
    if (is_checkout()) {
    ?>
        <style>
            /* Make the phone field wrapper full width */
            .iti.iti--allow-dropdown {
                width: 100%;
            }

            html[dir="rtl"] .iti__tel-input:not(#custom-whatsapp-modal-5987845-content .iti__tel-input) {
                text-align: right !important;
            }
        </style>
<?php
    }
});



add_action('woocommerce_checkout_update_user_meta', function ($user_id, $posted) {
    $whatsapp_cart_abandon_visible = get_option('whatsapp_notification_settings')['whatsapp_cart_abandon_visible'] ?? 'false';  // Using the null coalescing operator to avoid undefined index notice.
    if ($whatsapp_cart_abandon_visible == 'true') {
        if (!empty($_POST['whatsapp_full_number'])) {
            $user_verify_phone_time = gmdate('Y-m-d H:i:s');
            $fullPhoneNumber = sanitize_text_field($_POST['whatsapp_full_number']);
            $fullPhoneNumber = WhatsAppNotifier::fixPhoneNumber($fullPhoneNumber);
            $exist_user = WhatsAppUsers::exists($fullPhoneNumber);
            if ($exist_user == false) {
                $user_verify_phone_time = gmdate('Y-m-d H:i:s');
                WhatsAppUsers::create_or_update($fullPhoneNumber, [
                    'phone_verified' => 0,
                    'user_verify_phone_time' => $user_verify_phone_time,
                    'got_verification_coupon' => 0,
                    'got_cart_coupon' => 0,
                    'coupon_added_to_cart' => 0,
                    'phone' => $fullPhoneNumber,
                    'country_code' => sanitize_text_field($_POST['whatsapp_country_code']),
                ]);
            }
        }
    }
}, 10, 2);

register_activation_hook(__FILE__, 'whatsapp_create_custom_table');

function whatsapp_create_custom_table()
{
    global $wpdb;
    $table_name = $wpdb->prefix . 'whatsapp_notification_users';
    $charset_collate = $wpdb->get_charset_collate();

    $sql1 = "CREATE TABLE $table_name (
        id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        phone VARCHAR(20) NOT NULL UNIQUE,
        country_code VARCHAR(5) NOT NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        phone_verified TINYINT(1) DEFAULT 0,
        user_verify_phone_time DATETIME DEFAULT CURRENT_TIMESTAMP,
        got_cart_coupon TINYINT(1) DEFAULT 0,
        got_verification_coupon TINYINT(1) DEFAULT 0,
        coupon_code VARCHAR(20) NULL,
        coupon_added_to_cart TINYINT(1) DEFAULT 0,
        PRIMARY KEY (id)
    ) $charset_collate;";

    // 🔔 New newsletter log table
    $newsletters_table = $wpdb->prefix . 'whatsapp_newsletters_sent';
    $sql2 = "CREATE TABLE $newsletters_table (
        id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        phone VARCHAR(20) NOT NULL,
        message TEXT NOT NULL,
        sent_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        campaign_name VARCHAR(100) DEFAULT NULL,
        success TINYINT(1) DEFAULT 1,
        error TEXT DEFAULT NULL,
        PRIMARY KEY (id),
        INDEX(phone),
        INDEX(sent_at)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql1);
    dbDelta($sql2);
}
